<?php
class Tax extends CI_Model
{
	public function __construct(){
        parent::__construct();
        $this->db->query("SET sql_mode=(SELECT REPLACE(@@sql_mode, 'ONLY_FULL_GROUP_BY', ''));");
    }
	/*
	Determines if a given tax_code is on file
	*/
	public function exists($tax_code)
	{
		$this->db->from('tax_codes');
		$this->db->where('tax_code', $tax_code);

		return ($this->db->get()->num_rows() == 1);
	}

	public function tax_rate_exists($tax_code, $tax_category_id)
	{
		$this->db->from('tax_code_rates');
		$this->db->where('rate_tax_code', $tax_code);
		$this->db->where('rate_tax_category_id', $tax_category_id);

		return ($this->db->get()->num_rows() == 1);
	}

	/*
	Gets total of rows
	*/
	public function get_total_rows()
	{
		$this->db->from('tax_codes');

		return $this->db->count_all_results();
	}

	/*
	Gets information about a particular tax_code
	*/
	public function get_info($tax_code)
	{
		$this->db->from('tax_codes');
		$this->db->join('tax_code_rates',
			'tax_code = rate_tax_code and rate_tax_category_id = 0', 'LEFT');
		$this->db->join('tax_categories',
			'rate_tax_category_id = tax_category_id');
		$this->db->where('tax_code', $tax_code);

		$query = $this->db->get();

		if($query->num_rows() == 1)
		{
			return $query->row();
		}
		else
		{
			//Get empty base parent object
			$tax_code_obj = new stdClass();

			//Get all the fields from tax_codes table
			foreach($this->db->list_fields('tax_codes') as $field)
			{
				$tax_code_obj->$field = '';
			}
			foreach($this->db->list_fields('tax_code_rates') as $field)
			{
				$tax_code_obj->$field = '';
			}

			return $tax_code_obj;
		}
	}

	/*
Gets information about a particular tax_code
*/
	public function get_rate_info($tax_code, $tax_category_id)
	{
		$this->db->from('tax_code_rates');
		$this->db->join('tax_categories',
			'rate_tax_category_id = tax_category_id');
		$this->db->where('rate_tax_code', $tax_code);
		$this->db->where('rate_tax_category_id', $tax_category_id);

		$query = $this->db->get();

		if($query->num_rows() == 1)
		{
			return $query->row();
		}
		else
		{
			//Get empty base parent object
			$tax_rate_obj = new stdClass();

			//Get all the fields from tax_codes table
			foreach($this->db->list_fields('tax_code_rates') as $field)
			{
				$tax_rate_obj->$field = '';
			}
			//Get all the fields from tax_code_rates table
			foreach($this->db->list_fields('tax_categories') as $field)
			{
				$tax_rate_obj->$field = '';
			}

			return $tax_rate_obj;
		}
	}

	/*
	 * Gets the tax code to use for a given customer
	 */
	public function get_sales_tax_code($city = '', $state = '')
	{
		// if tax code using both city and state cannot be found then  try again using just the state
		// if the state tax code cannot be found then try again using blanks for both
		$this->db->from('tax_codes');
		$this->db->where('city', $city);
		$this->db->where('state', $state);
		$this->db->where('tax_code_type', '0'); // sales tax

		$query = $this->db->get();

		if($query->num_rows() == 1)
		{
			return $query->row()->tax_code;
		}
		else
		{
			$this->db->from('tax_codes');
			$this->db->where('city', '');
			$this->db->where('state', $state);
			$this->db->where('tax_code_type', '0'); // sales tax

			$query = $this->db->get();

			if($query->num_rows() == 1)
			{
				return $query->row()->tax_code;
			}
			else
			{
				return $this->config->item('default_origin_tax_code');
			}
		}

		return FALSE;
	}

	/*
	Inserts or updates a tax_codes entry
	*/
	public function save(&$tax_code_data, $tax_rate_data, $tax_code = -1)
	{
		if(!$this->exists($tax_code))
		{
			if($this->db->insert('tax_codes', $tax_code_data))
			{
				$this->save_tax_rates($tax_rate_data, $tax_code);
				return TRUE;
			}

			return FALSE;
		}

		$this->db->where('tax_code', $tax_code);
		if ($this->db->update('tax_codes', $tax_code_data))
		{
			$this->save_tax_rates($tax_rate_data, $tax_code);
			return TRUE;
		}
		else
		{
			return  FALSE;
		}
	}

	public function save_tax_rates(&$tax_rate_data, $tax_code)
	{
		if(!$this->tax_rate_exists($tax_code, $tax_rate_data['rate_tax_category_id']))
		{
			if($this->db->insert('tax_code_rates', $tax_rate_data))
			{
				return TRUE;
			}

			return FALSE;
		}

		$this->db->where('rate_tax_code', $tax_code);
		$this->db->where('rate_tax_category_id', $tax_rate_data['rate_tax_category_id']);

		return $this->db->update('tax_code_rates', $tax_rate_data);
	}

	/*
	Inserts or updates an item kit's items
	*/
	public function save_tax_rate_exceptions(&$tax_rate_data, $tax_code)
	{
		$success = TRUE;

		//Run these queries as a transaction, we want to make sure we do all or nothing

		$this->db->trans_start();

		// Delete all exceptions for the given tax_code
		$this->delete_tax_rate_exceptions($tax_code);

		if ($tax_rate_data != NULL)
		{
			foreach ($tax_rate_data as $row) 
			{
				$row['rate_tax_code'] = $tax_code;
				$success &= $this->db->insert('tax_code_rates', $row);
			}
		}

		$this->db->trans_complete();

		$success &= $this->db->trans_status();

		return $success;
	}

	/*
	Deletes one tax_codes entry
	*/
	public function delete($tax_code)
	{
		return $this->db->delete('tax_codes', array('tax_code' => $tax_code));
	}

	/*
	Deletes a list of tax codes
	*/
	public function delete_list($tax_codes)
	{
		$this->db->where_in('tax_code', $tax_codes);

		return $this->db->delete('tax_codes');
	}


	/*
	Deletes all tax_rate_exceptions for given tax codes
	*/
	public function delete_tax_rate_exceptions($tax_code)
	{
		$this->db->where('rate_tax_code', $tax_code);
		$this->db->where('rate_tax_category_id !=', 0);

		return $this->db->delete('tax_code_rates');
	}

	/*
	Performs a search on tax_codes
	*/
	public function search($search, $rows = 0, $limit_from = 0, $sort = 'tax_code', $order = 'asc')
	{
		$this->db->from('tax_codes');
		$this->db->join('tax_code_rates',
			'tax_code = rate_tax_code and rate_tax_category_id = 0', 'LEFT');
		if (!empty($search))
		{
			$this->db->like('tax_code', $search);
			$this->db->or_like('tax_code_name', $search);
		}
		$this->db->order_by($sort, $order);

		if($rows > 0)
		{
			$this->db->limit($rows, $limit_from);
		}

		return $this->db->get();
	}

	/*
	Gets tax_codes
	*/
	public function get_found_rows($search)
	{
		$this->db->from('tax_codes');
		if (!empty($search))
		{
			$this->db->like('tax_code', $search);
			$this->db->or_like('tax_code_name', $search);
		}

		return $this->db->get()->num_rows();
	}

	public function get_tax_code_type_name($tax_code_type) {
		if ($tax_code_type == '0')
		{
			return $this->lang->line('taxes_sales_tax');
		}
		else
		{
			return $this->lang->line('taxes_vat_tax');
		}
	}

	public function get_sales_tax_codes_search_suggestions($search, $limt = 99999)
	{

		$suggestions = array();

		$this->db->from('tax_codes');
		if (!empty($search))
		{
			$this->db->like('tax_code', $search);
			$this->db->or_like('tax_code_name', $search);
		}
		$this->db->order_by('tax_code_name', 'asc');

		foreach($this->db->get()->result() as $row)
		{
			$suggestions[] = array('value' => $row->tax_code, 'label' => ($row->tax_code . ' ' . $row->tax_code_name));
		}

		//only return $limit suggestions
		if(count($suggestions > $limit))
		{
			$suggestions = array_slice($suggestions, 0,$limit);
		}

		return $suggestions;
	}

	public function get_tax_category_suggestions($search)
	{
		$suggestions = array();

		$this->db->from('tax_categories');
		$this->db->where('tax_category_id !=', 0);
		if (!empty($search))
		{
			$this->db->like('tax_category', '%'.$search.'%');
		}
		$this->db->order_by('tax_category', 'asc');

		foreach($this->db->get()->result() as $row)
		{
			$suggestions[] = array('value' => $row->tax_category_id, 'label' => $row->tax_category);
		}

		return $suggestions;
	}

	public function get_tax_category($tax_category_id)
	{
		$this->db->select('tax_category');
		$this->db->from('tax_categories');
		$this->db->where('tax_category_id', $tax_category_id);

		return $this->db->get()->row()->tax_category;
	}

	public function get_all_tax_categories()
	{
		$this->db->from('tax_categories');
		$this->db->order_by('tax_category_id');

		return $this->db->get();
	}

	public function get_tax_category_id($tax_category)
	{
		$this->db->select('tax_category_id');
		$this->db->from('tax_categories');

		return $this->db->get()->row()->tax_category_id;
	}

	public function get_tax_code_rate_exceptions($tax_code)
	{
		$this->db->from('tax_code_rates');
		$this->db->join('tax_categories', 'rate_tax_category_id = tax_category_id');
		$this->db->where('rate_tax_code', $tax_code);
		$this->db->where('rate_tax_category_id !=', 0);
		$this->db->order_by('tax_category', 'asc');

		return $this->db->get()->result_array();
	}
}
?>
